<?php
/**
 * Plugin Name: Elementor Davor Visibility Control
 * Description: Adds visibility control to Elementor widgets based on the user's "davor" meta field, and includes an endpoint to update the "davor" field for a specific user.
 * Version: 1.2
 * Author: Your Name
 * Text Domain: elementor-davor-visibility
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

// Check if Elementor is active
add_action( 'plugins_loaded', 'edvc_check_elementor' );

function edvc_check_elementor() {
    if ( ! did_action( 'elementor/loaded' ) ) {
        add_action( 'admin_notices', 'edvc_elementor_not_active' );
    }
}

function edvc_elementor_not_active() {
    echo '<div class="notice notice-warning is-dismissible">
        <p><strong>Elementor Davor Visibility Control</strong> requires Elementor to be installed and activated.</p>
    </div>';
}

// Add visibility controls to Elementor widgets
add_action( 'elementor/element/after_section_end', 'edvc_add_visibility_controls', 10, 3 );

function edvc_add_visibility_controls( $element, $section_id, $args ) {
    // Add the controls only to widgets, not sections or columns
    if ( 'section' !== $element->get_name() && 'column' !== $element->get_name() ) {
        if ( 'section_custom_css_pro' === $section_id ) { // Ensure we add after the advanced section
            $element->start_controls_section(
                'edvc_visibility_section',
                [
                    'label' => __( 'Davor Visibility', 'elementor-davor-visibility' ),
                    'tab'   => \Elementor\Controls_Manager::TAB_ADVANCED,
                ]
            );

            $element->add_control(
                'edvc_enable_visibility',
                [
                    'label'        => __( 'Enable Visibility Control', 'elementor-davor-visibility' ),
                    'type'         => \Elementor\Controls_Manager::SWITCHER,
                    'label_on'     => __( 'Yes', 'elementor-davor-visibility' ),
                    'label_off'    => __( 'No', 'elementor-davor-visibility' ),
                    'return_value' => 'yes',
                    'default'      => '',
                ]
            );

            $element->add_control(
                'edvc_condition',
                [
                    'label'     => __( 'Condition', 'elementor-davor-visibility' ),
                    'type'      => \Elementor\Controls_Manager::SELECT,
                    'default'   => 'has_value',
                    'options'   => [
                        'has_value'     => __( 'User has value', 'elementor-davor-visibility' ),
                        'not_has_value' => __( 'User does not have value', 'elementor-davor-visibility' ),
                    ],
                    'condition' => [
                        'edvc_enable_visibility' => 'yes',
                    ],
                ]
            );

            $element->add_control(
    'edvc_meta_value_1',
    [
        'label'       => __( 'Meta Value 1', 'elementor-davor-visibility' ),
        'type'        => \Elementor\Controls_Manager::TEXT,
        'default'     => '',
        'placeholder' => __( 'Enter first meta value', 'elementor-davor-visibility' ),
        'condition'   => [
            'edvc_enable_visibility' => 'yes',
        ],
    ]
);

$element->add_control(
    'edvc_meta_value_2',
    [
        'label'       => __( 'Meta Value 2', 'elementor-davor-visibility' ),
        'type'        => \Elementor\Controls_Manager::TEXT,
        'default'     => '',
        'placeholder' => __( 'Enter second meta value', 'elementor-davor-visibility' ),
        'condition'   => [
            'edvc_enable_visibility' => 'yes',
        ],
    ]
);


            $element->end_controls_section();
        }
    }
}

// Control widget rendering based on user meta field
add_action( 'elementor/widget/render_content', 'edvc_control_widget_visibility', 10, 2 );

function edvc_control_widget_visibility( $content, $widget ) {
    $settings = $widget->get_settings_for_display();

    if ( isset( $settings['edvc_enable_visibility'] ) && 'yes' === $settings['edvc_enable_visibility'] ) {
        $condition      = $settings['edvc_condition'];
        $meta_value_1   = sanitize_text_field( $settings['edvc_meta_value_1'] );
        $meta_value_2   = sanitize_text_field( $settings['edvc_meta_value_2'] );
        $user_value     = '';

        if ( is_user_logged_in() ) {
            $user_value = get_user_meta( get_current_user_id(), 'davor', true );
        }

        $should_render = false;

        // Adjusted condition checks
        if ( 'has_value' === $condition && ( $user_value === $meta_value_1 || $user_value === $meta_value_2 ) ) {
            $should_render = true;
        } elseif ( 'not_has_value' === $condition && ( empty( $user_value ) || ( $user_value !== $meta_value_1 && $user_value !== $meta_value_2 ) ) ) {
            $should_render = true;
        }

        if ( ! $should_render ) {
            return ''; // Do not render the widget if the condition is not met
        }
    }

    return $content;
}

// Register the REST API route to update 'davor' meta field for a specific user
add_action( 'rest_api_init', function () {
    register_rest_route( 'custom/v1', '/update_davor/(?P<id>\d+)', array(
        'methods'  => 'POST',
        'callback' => 'update_davor_meta_for_user',
        'permission_callback' => '__return_true', // Allow access without authentication
    ));
});

/**
 * Callback to handle the POST request to update 'davor' meta field for a specific user by ID
 */
function update_davor_meta_for_user( $request ) {
    $user_id = (int) $request->get_param( 'id' );
    
    // Check if user exists
    if ( ! get_user_by( 'ID', $user_id ) ) {
        return new WP_Error( 'no_user', 'User does not exist', array( 'status' => 404 ) );
    }

    // Check if 'davor_value' is provided in the request
    $davor_value = sanitize_text_field( $request->get_param( 'davor_value' ) );
    if ( empty( $davor_value ) ) {
        return new WP_Error( 'no_value', 'No value provided for "davor"', array( 'status' => 400 ) );
    }

    // Update the user meta field 'davor' for the specified user ID
    update_user_meta( $user_id, 'davor', $davor_value );

    return rest_ensure_response( array(
        'status'  => 'success',
        'message' => 'Davor meta field updated successfully for user ID ' . $user_id,
        'user_id' => $user_id,
        'davor'   => $davor_value
    ));
}